using Microsoft.Data.SqlClient;
using QTCProject.Models;
using System.Text;
using System.Text.Json;

public class PatientDataAccess
{
    private readonly string _connectionString;
    private readonly RSACrypt _rsa;
    private readonly Vault _vault;

    private readonly HttpClient _httpClient;

    /** Creates an instance of the PatientDataAccess class
    
        @parameter connectionString: string representaion of the URI that is used to connect to a Microsoft SQL Database
    */
    public PatientDataAccess(string connectionString)
    {
        _connectionString = connectionString;
        _rsa = new RSACrypt();
        _vault = new Vault();

        _httpClient = new HttpClient();
    }

    /** Adds a patient to a Microsoft SQL Database

        @parameter patient: Patient object holding the patient information to be stored
    */
    public void AddPatient(Patient patient)
    {
        // Retrieve AES key and IV from Azure Key Vault
        string stringKeyIV = _vault.GetSessionKey(_vault.GetCurrentKey());
        string[] split = stringKeyIV.Split(",");
        (byte[] aesKey, byte[] aesIV) = (Convert.FromBase64String(split[0]), Convert.FromBase64String(split[1]));

        // Encrypt patient's SSN
        patient.SSN = Convert.ToBase64String(AESCrypt.Encrypt(patient.SSN, aesKey, aesIV));

        // Retrieve public key from vault
        // Encrypt session key with public key, and set patient's SessionKey to encrypted session key
        string publicKey = _vault.GetPublicOrPrivateKey("publicRSA");
        patient.SessionKey = _rsa.Encrypt(stringKeyIV, publicKey);
        
        // Store encrypted data in the database
        using (SqlConnection conn = new SqlConnection(_connectionString))
        {
            conn.Open();
            string query = @"
                INSERT INTO QTC 
                (FirstName, LastName, SSN, DOB, MiddleInitial, Gender, Email, CellPhone, SessionKey) 
                VALUES (@FirstName, @LastName, @SSN, @DOB, @MiddleInitial, @Gender, @Email, @CellPhone, @SessionKey)";

            using (SqlCommand cmd = new SqlCommand(query, conn))
            {
                cmd.Parameters.AddWithValue("@FirstName", (object) patient.FirstName ?? DBNull.Value);
                cmd.Parameters.AddWithValue("@LastName", (object) patient.LastName ?? DBNull.Value);
                cmd.Parameters.AddWithValue("@SSN", (object) patient.SSN ?? DBNull.Value);
                cmd.Parameters.AddWithValue("@DOB", (object) DateTime.Parse(patient.DOB) ?? DBNull.Value);
                cmd.Parameters.AddWithValue("@MiddleInitial", (object) patient.MiddleInitial ?? DBNull.Value);
                cmd.Parameters.AddWithValue("@Gender", (object) patient.Gender ?? DBNull.Value);
                cmd.Parameters.AddWithValue("@Email", (object) patient.Email ?? DBNull.Value);
                cmd.Parameters.AddWithValue("@CellPhone", (object) patient.CellPhone ?? DBNull.Value);
                cmd.Parameters.AddWithValue("@SessionKey", (object) patient.SessionKey ?? DBNull.Value);
                cmd.ExecuteNonQuery();
            }
        }
    }



    /** Creates and sets query used for finding patients that contains the search string and returns the records

        @parameter searchString: query string that is used to retrieve a set of patients

        @return patients: a list of patient objects holding information from each record that is returned from the database
    */
    public List<Patient> SearchPatient(string searchString)
    {
        List<Patient> patients = new List<Patient>();

        using (SqlConnection conn = new SqlConnection(_connectionString))
        {
            conn.Open();
            string query = @"
                SELECT * FROM QTC
                WHERE MemberId LIKE @searchString
                OR FirstName LIKE @searchString
                OR LastName LIKE @searchString";

            using (SqlCommand cmd = new SqlCommand(query, conn))
            {
                cmd.Parameters.AddWithValue("@searchString", $"%{searchString}%");
                FormatSQLToPatients(cmd, patients);
            }
        }
        return patients;
    }

    /** Dynamically creates and sets query according to the number of selected patients to retrieve from the database and returns the records

        @parameter selectedPatients: string array of patient ID's used to retrieve patients

        @return patients: a list of patient objects holding information from each record that is returned from the database
    */
    private List<Patient> RetrievePatients(string[] selectedPatients)
    {
        List<Patient> patients = new List<Patient>();

        using (SqlConnection conn = new SqlConnection(_connectionString))
        {
            conn.Open();
            StringBuilder query = new StringBuilder(@"SELECT * FROM QTC WHERE MemberId in (");
            for (int i = 0; i < selectedPatients.Length - 1; i++) query.Append("@value" + i + ",");
            query.Append("@value" + (selectedPatients.Length - 1) + ")");

            using (SqlCommand cmd = new SqlCommand(query.ToString(), conn))
            {
                for (int i = 0; i < selectedPatients.Length; i++) cmd.Parameters.AddWithValue("@value" + i, selectedPatients[i]);
                FormatSQLToPatients(cmd, patients);
            }
        }
        return patients;
    }

    /** Helper function that retrieves records from database according to the passed query, formats the returned records into Patient objects, and adds them to a list

        @parameter cmd: SQL command that executes the set query
        @parameter patients: list of patients that records are added to after being formatted to Patient objects
    */
    private void FormatSQLToPatients(SqlCommand cmd, List<Patient> patients)
    {
        using (SqlDataReader reader = cmd.ExecuteReader())
        {
            while (reader.Read())
            {
                Patient patient = new Patient
                {
                    MemberId = reader["MemberId"].ToString(),
                    FirstName = reader["FirstName"].ToString(),
                    LastName = reader["LastName"].ToString(),
                    SSN = reader["SSN"].ToString(),
                    DOB = ((DateTime)reader["DOB"]).ToString("yyyy-MM-dd"),
                    MiddleInitial = reader["MiddleInitial"].ToString(),
                    Gender = reader["Gender"].ToString(),
                    Email = reader["Email"].ToString(),
                    CellPhone = reader["CellPhone"].ToString(),
                    SessionKey = reader["SessionKey"].ToString()
                };
                patients.Add(patient);
            }
        }
    }

    /** Encrypts information retrieved from the database (selected from HTML form) and sends to the selected endpoint

        @parameter selectedPatients: array of strings holding patient IDs used to retrieve from the vault
        @parameter action: string that determines if the information is sent to an internal or external application
    */
    public async Task SendByDatabase(string[] selectedPatients, string action) 
    {
        var publicClientKey = action == "SendExternal" ? _vault.GetPublicOrPrivateKey("App4Public") : null;
        string privateQTCkey = _vault.GetPublicOrPrivateKey("privateRSA");

        List<Patient> patientsList = RetrievePatients(selectedPatients);

        foreach(Patient patient in patientsList)
        {
            string stringKeyIV = _rsa.Decrypt(patient.SessionKey, privateQTCkey);
            string[] splitKeyIV = stringKeyIV.Split(",");
            (byte[] aesKey, byte[] aesIV) = (Convert.FromBase64String(splitKeyIV[0]), Convert.FromBase64String(splitKeyIV[1]));

            foreach (var prop in patient.GetType().GetProperties().Where(p => p.Name != "SSN" && p.Name != "SessionKey"))
            {
                var value = prop.GetValue(patient) == null ?  null : Convert.ToBase64String(AESCrypt.Encrypt(prop.GetValue(patient).ToString(), aesKey, aesIV));
                prop.SetValue(patient, value);
            }

            if (action == "SendExternal") patient.SessionKey = _rsa.Encrypt(stringKeyIV, publicClientKey);
        }

        await SendToEndpoint(patientsList, action);
    }

    /** Encrypts information entered into HTML form without storing to the database and sends to the selected endpoint

        @parameter patient: Patient object holding the information that is to be encrypted and sent
        @parameter action: string that determines if the information is sent to an internal or external application
    */
    public async Task SendByForm(Patient patient, string action)
    {
        // Retreive AES key from vault
        string stringKeyIV = _vault.GetSessionKey(_vault.GetCurrentKey());
        string[] splitKeyIV = stringKeyIV.Split(",");
        (byte[] aesKey, byte[] aesIV) = (Convert.FromBase64String(splitKeyIV[0]), Convert.FromBase64String(splitKeyIV[1]));
        List<Patient> patientsList = new List<Patient>();

        // Loop through patient properties and encrypt each value with AES key
        foreach (var prop in patient.GetType().GetProperties())
        {
            var value = prop.GetValue(patient) == null ?  null : Convert.ToBase64String(AESCrypt.Encrypt(prop.GetValue(patient).ToString(), aesKey, aesIV));
            prop.SetValue(patient, value);
        }
        patientsList.Add(patient);

        // Retrive internal/external public key and set patient's session key to the encrypted AES key
        string publicKey = action == "SendInternal" ? _vault.GetPublicOrPrivateKey("publicRSA") : _vault.GetPublicOrPrivateKey("App4Public");
        patient.SessionKey = _rsa.Encrypt(stringKeyIV, publicKey);

        await SendToEndpoint(patientsList, action);
    }

    /** Helper function that sends ciphertext to the specified endpoint

        @parameter patientsList: list of patient objects selected in HTML form
        @parameter action: string that determines if the information is sent to an internal or external application
    */
    private async Task SendToEndpoint(List<Patient> patientsList, string action)
    {
        var patientsJSON = JsonSerializer.Serialize(patientsList, new JsonSerializerOptions { WriteIndented = true });
        var response = action == "SendInternal" 
            ? await _httpClient.PostAsync("http://localhost:5197/api/API", new StringContent(patientsJSON, Encoding.UTF8, "application/json")) 
            : await _httpClient.PostAsync("external-application-url", new StringContent(patientsJSON, Encoding.UTF8, "application/json"));

        // Print whether data was sent successfully
        var type = action == "SendInternal" ? "internal" : "external";
        var status = response.IsSuccessStatusCode == true 
            ? "Data sent to " + type + " application successfully" 
            : "Failed to send data to " + type + " application. Status code: {response.StatusCode}";
        Console.WriteLine(status);
    }
}