//use RSA encryption algorithm to encrypt and decrypt data
//use the built-in .Net cryptography library
//input will take the SSN and output will be the encrypted SSN

using System.Security.Cryptography;
using System.Text;
using System.Xml.Serialization;

public class RSACrypt
{
    /** Generates RSA public and private keys

        @return privateKeyXML: string representation of XML-formatted RSA private key
        @return publicKeyXML: string representation of XML-formatted RSA public key
    */
    public (string privateKey, string publicKey) GenerateKeys()
    {
        using (RSA rsa = RSA.Create())
        {
            RSAParameters privateKeyParams = rsa.ExportParameters(true);
            RSAParameters publicKeyParams = rsa.ExportParameters(false);

            string privateKeyXML = SerializeKey(privateKeyParams);
            string publicKeyXML = SerializeKey(publicKeyParams);

            // Set the expiry time for keys, e.g., 24 hours from now
            return (privateKeyXML, publicKeyXML);
        }
    }

    /** Encrypts plaintext data using the public key

        @parameter plaintext: string data to encrypt
        @parameter publicKeyXML: RSA public that that must be in XML document format

        @return encryptedData: string representation of ciphertext
    */
    public string Encrypt(string plainText, string publicKeyXML)
    {
        using (RSA rsa = RSA.Create())
        {
            rsa.ImportParameters(DeserializeKey(publicKeyXML));
            byte[] dataToEncrypt = Encoding.UTF8.GetBytes(plainText);
            byte[] encryptedData = rsa.Encrypt(dataToEncrypt, RSAEncryptionPadding.OaepSHA256);
            return Convert.ToBase64String(encryptedData);
        }
    }

    /** Decrypts ciphertext data using an RSA public key

        @parameter cipherText: string data to decrypt
        @parameter privateKeyXML: RSA private key that that must be in XML document format

        @return decryptedData: string representation of plaintext
    */
    public string Decrypt(string cipherText, string privateKeyXML)
    {
        using (RSA rsa = RSA.Create())
        {
            rsa.ImportParameters(DeserializeKey(privateKeyXML));
            byte[] dataToDecrypt = Convert.FromBase64String(cipherText);
            byte[] decryptedData = rsa.Decrypt(dataToDecrypt, RSAEncryptionPadding.OaepSHA256);
            return Encoding.UTF8.GetString(decryptedData);
        }
    }

    /** Formats key into XML document format

        @parameter key: key to format

        @return sw: string representaion of XML-formatted key
    */
    private string SerializeKey(RSAParameters key)
    {
        var sw = new StringWriter();
        new XmlSerializer(typeof(RSAParameters)).Serialize(sw, key);
        return sw.ToString();
    }

    /** Formats XML-formatted key into RSAParameter object

        @parameter keyXML: string representation of XML-formatted key

        @return sr: RSAParameter object with key's values
    */
    private RSAParameters DeserializeKey(string keyXml)
    {
        using (var sr = new StringReader(keyXml))
        {
            return (RSAParameters)new XmlSerializer(typeof(RSAParameters)).Deserialize(sr)!;
        }
    }
}